/*
 * mods_tegradc.c - This file is part of NVIDIA MODS kernel driver.
 *
 * Copyright (c) 2014, NVIDIA CORPORATION.  All rights reserved.
 *
 * NVIDIA MODS kernel driver is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * NVIDIA MODS kernel driver is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NVIDIA MODS kernel driver.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/uaccess.h>
#include <mach/dc.h>
#include <../drivers/video/tegra/dc/dc_priv.h>
#include "mods_internal.h"

#ifdef CONFIG_TEGRA_ISOMGR
static void mods_tegra_dc_set_windowattr_basic(struct tegra_dc_win *win,
		       const struct MODS_TEGRA_DC_WINDOW *mods_win)
{
	win->global_alpha = 0;
	win->z            = 0;
	win->stride       = 0;
	win->stride_uv    = 0;

	win->flags = TEGRA_WIN_FLAG_ENABLED;
	if (mods_win->flags & MODS_TEGRA_DC_WINDOW_FLAG_TILED)
		win->flags |= TEGRA_WIN_FLAG_TILED;
#if defined(CONFIG_TEGRA_DC_SCAN_COLUMN)
	if (mods_win->flags & MODS_TEGRA_DC_WINDOW_FLAG_SCAN_COL)
		win->flags |= TEGRA_WIN_FLAG_SCAN_COLUMN;
#endif

	win->fmt = mods_win->pixformat;
	win->x.full = mods_win->x;
	win->y.full = mods_win->y;
	win->w.full = mods_win->w;
	win->h.full = mods_win->h;
	/* XXX verify that this doesn't go outside display's active region */
	win->out_x = mods_win->out_x;
	win->out_y = mods_win->out_y;
	win->out_w = mods_win->out_w;
	win->out_h = mods_win->out_h;

	mods_debug_printk(DEBUG_TEGRADC,
		"mods_tegra_dc_set_windowattr_basic window %u:\n"
		"\tflags : 0x%08x\n"
		"\tfmt   : %u\n"
		"\tinput : (%u, %u, %u, %u)\n"
		"\toutput: (%u, %u, %u, %u)\n",
		win->idx, win->flags, win->fmt, dfixed_trunc(win->x),
		dfixed_trunc(win->y), dfixed_trunc(win->w),
		dfixed_trunc(win->h), win->out_x, win->out_y, win->out_w,
		win->out_h);
}
#endif

int esc_mods_tegra_dc_config_possible(struct file *fp,
				struct MODS_TEGRA_DC_CONFIG_POSSIBLE *args)
{
#ifndef CONFIG_TEGRA_ISOMGR
	return -EINVAL;
#else
	int ret;
	int i;
	struct tegra_dc *dc = tegra_dc_get_dc(args->head);
	struct tegra_dc_win *dc_wins[DC_N_WINDOWS];
	struct MODS_TEGRA_DC_WINDOW *mods_wins;
	LOG_ENT();

	BUG_ON(args->win_num > DC_N_WINDOWS);

	if (!dc) {
		LOG_EXT();
		return -EINVAL;
	}

	mods_wins = kzalloc(sizeof(*mods_wins) * args->win_num, GFP_KERNEL);

	if (copy_from_user(mods_wins, args->wins,
			sizeof(*mods_wins) * args->win_num)) {
		kfree(mods_wins);
		LOG_EXT();
		return -EFAULT;
	}

	for (i = 0; i < args->win_num; i++) {
		int idx = mods_wins[i].index;

		if (mods_wins[i].flags & MODS_TEGRA_DC_WINDOW_FLAG_ENABLED) {
			mods_tegra_dc_set_windowattr_basic(&dc->tmp_wins[idx],
							  &mods_wins[i]);
		} else {
			dc->tmp_wins[idx].flags = 0;
		}
		dc_wins[i] = &dc->tmp_wins[idx];
		mods_debug_printk(DEBUG_TEGRADC,
			"esc_mods_tegra_dc_config_possible head %u, "
			"using index %d for window %d\n",
			args->head, i, idx);
	}

	mods_debug_printk(DEBUG_TEGRADC,
		"esc_mods_tegra_dc_config_possible head %u, "
		"dc->mode.pclk %u\n",
		args->head, dc->mode.pclk);

	ret = tegra_dc_bandwidth_negotiate_bw(dc, dc_wins, args->win_num);
	args->possible = (ret == 0);
	for (i = 0; i < args->win_num; i++) {
		mods_wins[i].bandwidth = dc_wins[i]->new_bandwidth;
		mods_debug_printk(DEBUG_TEGRADC,
			"esc_mods_tegra_dc_config_possible head %u, "
			"window %d bandwidth %d\n",
			args->head, dc_wins[i]->idx, dc_wins[i]->new_bandwidth);
	}

	if (copy_to_user(args->wins, mods_wins,
		sizeof(*mods_wins) * args->win_num)) {
		kfree(mods_wins);
		LOG_EXT();
		return -EFAULT;
	}

	LOG_EXT();
	kfree(mods_wins);
	return 0;
#endif
}
